function segmentedImage = segmentImageLDAIter(imgPath, fgSample, bgSample)
%SEGMENTIMAGELDAIter Segment image using iterative LDA
%
%   Input:
%       imgPath : path to the input image
%       fgSample: pixels of foreground scribbles
%       bgSample: pixels of background scribbles
% 
%   Output:
%       segmentedImage: binary image representing the output segmentation
%
% (c) Moustafa Meshry, moustafa.meshry@alexu.edu.eg
%     Department of Compter and Systems Engineering, Alexandria University, Egypt.

    % extract features for all pixels
    imgFeatures = extractFeatures(imgPath, fgSample, bgSample);
    [height, width, featureVectorSize] = size(imgFeatures);
    
    % Create user fg/bg scribbles binary mask
    fgSampleMask = false(height, width);
    bgSampleMask = false(height, width);
    fgSampleMask(sub2ind(size(fgSampleMask), fgSample(:,1), fgSample(:,2))) = true;
    bgSampleMask(sub2ind(size(bgSampleMask), bgSample(:,1), bgSample(:,2))) = true;

    % constants for the iterative method
    kMaxIter = 5;
    kMaxTrainingSize = 8000;
    kGrowFactor = 1.3;

    iter = 1;
    while ((size(fgSample, 1) + size(bgSample, 1) < kMaxTrainingSize && ...
            iter <= kMaxIter) || iter == 1)
        iter = iter + 1;

        % select features of user-labeled foreground
        fgExpandedScribblesMask = false(height, width);
        fgExpandedScribblesMask(sub2ind([height, width], ...
            fgSample(:, 1), fgSample(:, 2))) = true;
        fgSampleMask3D = repmat(fgExpandedScribblesMask, 1, ...
            featureVectorSize);
        fgSampleMask3D = reshape(fgSampleMask3D, size(imgFeatures));
        fgFeatures = imgFeatures(fgSampleMask3D);
        fgFeatures = reshape(fgFeatures, [], featureVectorSize);

        % select features of user-labeled background
        bgExpandedScribblesMask = false(height, width);
        bgExpandedScribblesMask(sub2ind([height, width], ...
            bgSample(:, 1), bgSample(:, 2))) = true;
        bgSampleMask3D = repmat(bgExpandedScribblesMask, 1, ...
            featureVectorSize);
        bgSampleMask3D = reshape(bgSampleMask3D, size(imgFeatures));
        bgFeatures = imgFeatures(bgSampleMask3D);
        bgFeatures = reshape(bgFeatures, [], featureVectorSize);

        % LDA Training
        trainSample = [fgFeatures; bgFeatures];
        trainLabels = ones(size(trainSample, 1), 1);
        trainLabels(size(fgFeatures, 1) + 1 : end) = 0;

        W = LDA(trainSample, trainLabels);

        % LDA projection and classification
        pixelsVec = reshape(imgFeatures, height * width, []);
        pixelsVec = [ones(size(pixelsVec, 1), 1) pixelsVec];
        pixelsScores = pixelsVec * W';
                
        %% Classification Using LDA fg & bg Scores
        
        pixelsScores = reshape(pixelsScores, ...
            [height, width, size(pixelsScores, 2)]);
        bgMask = pixelsScores(:, :, 1) > pixelsScores(:, :, 2);
                
        %% Expanding scribbles using score difference between fg/bg
        
        scoresDiff = pixelsScores(:,:,2) - pixelsScores(:,:,1);
        [sortedVals, ~] = sort(scoresDiff(:));
        fgCutoffIndex = length(sortedVals) - floor(kGrowFactor * size(fgSample, 1)) + 1; 
        fgCutoffIndex = max(fgCutoffIndex, 1);
        bgCutoffIndex = floor(kGrowFactor * size(bgSample, 1)); 
        fgCutoffVal = sortedVals(fgCutoffIndex);
        bgCutoffIndex = min(bgCutoffIndex, length(sortedVals));
        bgCutoffVal = sortedVals(bgCutoffIndex);
        fgScoreMask = (scoresDiff > fgCutoffVal) | fgSampleMask;
        bgScoreMask = (scoresDiff < bgCutoffVal) | bgSampleMask;
        [y, x] = find(fgScoreMask);
        fgSample = [y x];
        [y, x] = find(bgScoreMask);
        bgSample = [y, x];        
    end
    
    [y, x] = find(bgMask);
    segmentedImage = repmat(255, height, width);
    segmentedImage(sub2ind(size(segmentedImage), y, x)) = 0;
    segmentedImage = postProcess(segmentedImage, fgSampleMask, bgSampleMask);
end

